package ga.view.streaming.nodes;

import ga.core.individual.IIndividual;

import com.jme3.bullet.PhysicsSpace;
import com.jme3.bullet.collision.PhysicsCollisionObject;
import com.jme3.bullet.collision.shapes.BoxCollisionShape;
import com.jme3.bullet.control.RigidBodyControl;
import com.jme3.bullet.joints.Point2PointJoint;
import com.jme3.math.Vector3f;
import com.jme3.scene.Node;

/**
 * This provides an anchor nodes that will pull attached nodes through the
 * scene.
 * 
 * @param <T>
 *          The generic type of the individuals.
 * 
 * @since 12.08.2012
 * @author Stephan Dreyer
 */
public class AnchorNode<T extends IIndividual<T>> extends Node {
  private final PhysicsSpace physicsSpace;

  private Point2PointJoint sliderJoint;
  private final RigidBodyControl control;
  private PanelNode<T> attachedNode;

  /**
   * Instantiates a new anchor node.
   * 
   * @param physicsSpace
   *          the physics space
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public AnchorNode(final PhysicsSpace physicsSpace) {
    super("Anchor Node");

    control = new RigidBodyControl(new BoxCollisionShape(new Vector3f(.5f, .4f,
        .5f)), 0);
    addControl(control);
    this.physicsSpace = physicsSpace;

    control.setKinematic(true);

    control.setCollisionGroup(PhysicsCollisionObject.COLLISION_GROUP_NONE);
    control.setCollideWithGroups(PhysicsCollisionObject.COLLISION_GROUP_NONE);

  }

  /**
   * Attachs a joint to the given node.
   * 
   * @param node
   *          the node
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public void attachJoint(final PanelNode<T> node) {
    this.attachedNode = node;
    attachedNode.setAnchor(this);
    sliderJoint = new Point2PointJoint(node.getControl(), control,
        Vector3f.ZERO, new Vector3f(-.5f, 0f, 0f));

    sliderJoint.setCollisionBetweenLinkedBodys(false);
    sliderJoint.setDamping(1f);
    sliderJoint.setImpulseClamp(5f);
    sliderJoint.setTau(.1f);

    physicsSpace.add(sliderJoint);
  }

  /**
   * Gets the control.
   * 
   * @return the control
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public RigidBodyControl getControl() {
    return control;
  }

  /**
   * Detaches the joint to the node.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public void detachJoint() {
    if (sliderJoint != null) {
      physicsSpace.remove(sliderJoint);
      sliderJoint.destroy();
      sliderJoint = null;
    }
  }

  @Override
  protected void setParent(final Node parent) {
    super.setParent(parent);
    control.setPhysicsLocation(this.getLocalTranslation());
  }

  /**
   * Gets the attached node.
   * 
   * @return the attached node
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public PanelNode<T> getAttachedNode() {
    return attachedNode;
  }
}
